/*:
 * @target MZ
 * @plugindesc v1.1 与ダメージ%ぶんのHP/MP/TPをライフスティール。装備・ステート・職業のメモ欄対応。
 * @author ChatGPT
 *
 * @help
 * ■概要
 * 与えた「HPダメージ」の○％ぶんだけ、攻撃者の HP/MP/TP を回復（吸収）します。
 * 率は装備・ステート・職業のメモ欄から取得し、合算（または最大値）します。
 *
 * ■メモ欄タグ
 * 使いやすいように2通りの記述を用意しています。装備、ステート、職業いずれにも書けます。
 *
 * 1) まとめて指定（カンマ区切り：HP,MP,TP）
 *    <LifeSteal: HP%, MP%, TP%>
 *    例) <LifeSteal:20, 5, 0>  … 与ダメの20%をHP回復、5%をMP回復、TPは0%
 *        <LifeSteal:10>        … HPのみ10%（MP/TPは0扱い）
 *
 * 2) 個別指定
 *    <LifeStealHP:20>
 *    <LifeStealMP:5>
 *    <LifeStealTP:3>
 *
 * まとめ指定と個別指定は**加算**されます（最終的に StackMode の規則で装備・ステート・職業ごとに集約）。
 *
 * ■合算規則
 * - 「装備の合計」「ステートの合計」「職業」それぞれで率を集約し、
 *   さらにそれらを StackMode に従って「sum（合計）」または「max（最大）」でまとめます。
 * - 既定は sum（全部足す）。
 *
 * ■挙動の詳細
 * - トリガー：攻撃行動で対象に与えたHPダメージ（>0）の直後。
 * - 吸収量：floor( 与HPダメージ × 各率% ) を攻撃者に付与（HP/MP/TP）。
 * - 0やマイナスダメージ、回復行為では発動しません。
 * - 攻撃者が戦闘不能なら回復しません。
 * - ポップアップはパラメータでON/OFF。
 *
 * ■例
 * 〈短剣（装備）〉          メモ欄: <LifeSteal:10>        → HP10%吸収
 * 〈毒の刃（ステート）〉     メモ欄: <LifeStealMP:3>       → MP3%吸収
 * 〈アサシン（職業）〉       メモ欄: <LifeSteal:0,0,5>     → TP5%吸収
 * 上記3つを満たすなら、合計で「HP10% + MP3% + TP5%」吸収（StackMode=sumのとき）。
 *
 * ■競合について
 * - Game_Action.apply に後がけでフックしています。他プラグインの同箇所書き換え順により
 *   表示順が変わる可能性があります。
 *
 * @param NoteTagBase
 * @text まとめ指定タグ名
 * @desc <タグ名:HP%,MP%,TP%> の形式。例: LifeSteal
 * @default LifeSteal
 *
 * @param NoteTagHP
 * @text 個別タグ名(HP)
 * @default LifeStealHP
 *
 * @param NoteTagMP
 * @text 個別タグ名(MP)
 * @default LifeStealMP
 *
 * @param NoteTagTP
 * @text 個別タグ名(TP)
 * @default LifeStealTP
 *
 * @param StackMode
 * @text 複数ソースの合算方法
 * @type select
 * @option 合計 (Sum)
 * @value sum
 * @option 最大 (Max)
 * @value max
 * @desc 装備/ステート/職業から集めた率のまとめ方。
 * @default sum
 *
 * @param ShowPopup
 * @text ポップアップ表示
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @default true
 *
 */
(() => {
  const pluginName = "DamageLifeStealEx";
  const p = PluginManager.parameters(pluginName);
  const TAG_BASE = String(p.NoteTagBase || "LifeSteal");
  const TAG_HP   = String(p.NoteTagHP   || "LifeStealHP");
  const TAG_MP   = String(p.NoteTagMP   || "LifeStealMP");
  const TAG_TP   = String(p.NoteTagTP   || "LifeStealTP");
  const STACK    = String(p.StackMode || "sum").toLowerCase();
  const SHOW     = String(p.ShowPopup || "true") === "true";

  /** ノートから {hp, mp, tp} 率を取り出す（まとめ＋個別） */
  function parseRatesFromNote(obj) {
    const r = { hp: 0, mp: 0, tp: 0 };
    if (!obj || !obj.meta) return r;

    // まとめ指定 <TAG_BASE: hp, mp, tp>
    if (obj.meta[TAG_BASE] !== undefined) {
      const parts = String(obj.meta[TAG_BASE]).split(",").map(s => Number(s.trim()));
      if (parts.length >= 1 && !Number.isNaN(parts[0])) r.hp += parts[0];
      if (parts.length >= 2 && !Number.isNaN(parts[1])) r.mp += parts[1];
      if (parts.length >= 3 && !Number.isNaN(parts[2])) r.tp += parts[2];
    }
    // 個別指定
    if (obj.meta[TAG_HP] !== undefined) {
      const v = Number(obj.meta[TAG_HP]);
      if (!Number.isNaN(v)) r.hp += v;
    }
    if (obj.meta[TAG_MP] !== undefined) {
      const v = Number(obj.meta[TAG_MP]);
      if (!Number.isNaN(v)) r.mp += v;
    }
    if (obj.meta[TAG_TP] !== undefined) {
      const v = Number(obj.meta[TAG_TP]);
      if (!Number.isNaN(v)) r.tp += v;
    }
    return r;
  }

  /** 複数率オブジェクトを sum / max でまとめる */
  function reduceRates(list) {
    if (list.length === 0) return { hp: 0, mp: 0, tp: 0 };
    if (STACK === "max") {
      return list.reduce((acc, v) => ({
        hp: Math.max(acc.hp, v.hp),
        mp: Math.max(acc.mp, v.mp),
        tp: Math.max(acc.tp, v.tp),
      }), { hp: 0, mp: 0, tp: 0 });
    }
    // sum
    return list.reduce((acc, v) => ({
      hp: acc.hp + v.hp,
      mp: acc.mp + v.mp,
      tp: acc.tp + v.tp,
    }), { hp: 0, mp: 0, tp: 0 });
  }

  /** バトラー（攻撃者）から装備/ステート/職業の率を集める */
  function gatherLifeStealRates(battler) {
    const packs = [];

    // 装備
    if (battler.equips) {
      const rates = [];
      for (const it of battler.equips()) {
        if (it) rates.push(parseRatesFromNote(it));
      }
      packs.push(reduceRates(rates));
    }

    // ステート
    if (battler.states) {
      const rates = battler.states().map(st => parseRatesFromNote(st));
      packs.push(reduceRates(rates));
    }

    // 職業（アクターのみ）
    if (battler.isActor && battler.isActor()) {
      const cls = battler.currentClass ? battler.currentClass() : null;
      if (cls) packs.push(parseRatesFromNote(cls));
    }

    // まとめ（装備/ステート/職業） → sum or max
    return reduceRates(packs);
  }

  // ダメージ適用後フック
  const _Game_Action_apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function(target) {
    _Game_Action_apply.call(this, target);

    const subject = this.subject();
    if (!subject || !target) return;

    const res = target.result();
    const hpDamage = res ? res.hpDamage : 0;

    if (hpDamage > 0 && subject.isAlive()) {
      const rate = gatherLifeStealRates(subject); // {hp, mp, tp}
      const healHP = Math.floor(hpDamage * (rate.hp || 0) / 100);
      const healMP = Math.floor(hpDamage * (rate.mp || 0) / 100);
      const healTP = Math.floor(hpDamage * (rate.tp || 0) / 100);

      if (healHP > 0) subject.gainHp(healHP);
      if (healMP > 0) subject.gainMp(healMP);
      if (healTP > 0) subject.gainTp(healTP);
      if (healHP > 0 || healMP > 0 || healTP > 0) {
        subject.refresh();

        if (SHOW) {
          // 攻撃者側のポップアップを一時的な result で表示
          const r = subject.result();
          const backup = {
            hpDamage: r.hpDamage, mpDamage: r.mpDamage, tpDamage: r.tpDamage, success: r.success
          };
          r.clear();
          r.success = true;
          r.hpDamage = -healHP;
          r.mpDamage = -healMP;
          r.tpDamage = -healTP;
          subject.startDamagePopup();

          // 値を戻す
          r.hpDamage = backup.hpDamage;
          r.mpDamage = backup.mpDamage;
          r.tpDamage = backup.tpDamage;
          r.success  = backup.success;
        }

        if ($gameParty.inBattle && $gameParty.inBattle()) {
          subject.performRecovery();
        }
      }
    }
  };
})();
